import 'package:flutter/cupertino.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:razinsoft_rider/core/routes/app_routes.dart';
import 'package:razinsoft_rider/core/state/app_state.dart';
import 'package:razinsoft_rider/core/utils/device_token_firebase.dart';
import 'package:razinsoft_rider/core/utils/helpers.dart';
import 'package:razinsoft_rider/data/models/reset_password_response.dart';
import 'package:razinsoft_rider/data/repositories/interfaces/auth_repo_interface.dart';
import 'package:razinsoft_rider/data/services/navigation_service.dart';
import 'package:razinsoft_rider/presentation/booking/provider/order_providers.dart';
import 'package:razinsoft_rider/presentation/profile/provider/rider_details_provider.dart';

import '../../../data/models/common_response.dart';
import '../../../data/models/login_response.dart';
import '../../../data/models/login_with_password_response.dart';
import '../../../data/models/logout_response.dart';
import '../../../data/models/otp_verify_response.dart';
import '../../../data/models/profile_update_response.dart';
import '../../../data/models/resend_otp_model/resend_otp_mode.dart';
import '../../../data/services/local_storage_service.dart';
import '../provider/auth_providers.dart';
import '../widgets/warning.dart';


class LoginNotifier extends StateNotifier<AppState<LoginResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;
  LoginNotifier({required this.ref, required this.authRepo}) : super(const AppState.initial());

  Future<void> login({required String phone, required String countryCode}) async {
    state = const AppState.loading();
    final String? deviceToken = await deviceTokenFirebase();
    await LocalStorageService().clearToken();
    final result = await authRepo.login(phone: phone, deviceToken: deviceToken, countryCode: countryCode);
    result.fold(
      (failure) {
        state = AppState.error(failure);
        showNotification( message: failure.message, );
      },
      (loginResponse) {
        _handleLoginSuccess(loginResponse, countryCode);
        return state = AppState.success(loginResponse);
      },
    );
  }

  void _handleLoginSuccess(LoginResponse loginResponse, String countryCode) {
    final localStorage = LocalStorageService();
    final isNewUser = loginResponse.data?.isNewRider == true;

    if (isNewUser) {
      showNotification(message: 'otp: ${loginResponse.data?.otp}', isSuccess: true);
      localStorage..savePhoneCode(countryCode)
      ..setRegistrationProgress(AppRoutes.verifyOtp);
      NavigationService.pushNamed(AppRoutes.verifyOtp, arguments: loginResponse.data?.otp );
    } else {
      NavigationService.pushNamed(AppRoutes.loginWithPasswordPage);
    }
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class LoginWithPassNotifier extends StateNotifier<AppState<LoginWithPasswordResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;

  LoginWithPassNotifier({required this.ref, required this.authRepo}) : super(const AppState.initial());

  Future<void> loginWithPassword({required String mobile, required String password, String? wantLogin}) async {
    state = const AppState.loading();
    final String? deviceToken = await deviceTokenFirebase();
    await LocalStorageService().clearToken();
    final result = await authRepo.loginWithPassword(mobile: mobile, password: password, deviceToken: deviceToken, wantLogin: wantLogin);
    result.fold(
      (failure) {
        showNotification(message: failure.message);
         state = AppState.error(failure);
      },
      (data) async{
        if(data.data?.otherDevice != null && data.data?.otherDevice == true){
          final bool? wantLogin = await showWarning();
          if(wantLogin == true){
            await loginWithPassword(mobile: mobile, password: password, wantLogin: 'yes');
          }else{
            NavigationService.pop();
            resetStateAfterDelay();
          }
          state = AppState.success(data);
          return;
        }
        await LocalStorageService().saveToken(data.data?.token ?? '');
        await LocalStorageService().saveUser(user: data.data?.user?.toJson() ?? {});
        LocalStorageService().setRegistrationProgress(AppRoutes.dashboard);
        ref.read(tripActivityNotifierProvider.notifier).checkTripActivity();
         state = AppState.success(data);

      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class ResendSignInNotifier extends StateNotifier<AppState<LoginWithPasswordResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;

  ResendSignInNotifier({required this.ref, required this.authRepo}) : super(const AppState.initial());

  Future<void> resendSignIn({required num? userId, required String? deviceToken}) async {
    state = const AppState.loading();
    final String? deviceToken = await deviceTokenFirebase();
    await LocalStorageService().clearToken();
    final result = await authRepo.resendSignIn(userId: userId, deviceToken: deviceToken);
    // final res = await authRepo.login(phone: phone, countryCode: countryCode)
    result.fold(
          (failure) {
        showNotification(message: failure.message);
        return state = AppState.error(failure);
      },
          (data) async{
        showNotification(message: data.message, isSuccess: true);

        await LocalStorageService().saveToken(data.data?.token ?? '');
        await LocalStorageService().saveUser(user: data.data?.user?.toJson() ?? {});
        LocalStorageService().setRegistrationProgress(AppRoutes.dashboard);
        await ref.read(tripActivityNotifierProvider.notifier).checkTripActivity();
        state = AppState.success(data);
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class OtpVerifyNotifier extends StateNotifier<AppState<OtpVerifyResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;
  OtpVerifyNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> verifyOtp({required String mobile, required String otp, String? wantLogin}) async {
    final String? deviceToken = await deviceTokenFirebase();
    final loginResponse = ref.read(loginNotifierProvider).maybeWhen(
          success: (data) => data,
          orElse: () => null,
        );

    state = const AppState.loading();
    final result = await authRepo.verifyOtp(mobile: mobile, otp: otp, deviceToken: deviceToken, wantLogin: wantLogin);
    result.fold(
      (failure) {
        showNotification(message: failure.message);
        state = AppState.error(failure);
      },
      (verifyResponse) async{
        if(verifyResponse.data?.otherDevice == true){
          final bool? wantLogin = await showWarning();
          if(wantLogin == true){
            await verifyOtp(mobile: mobile, otp: otp, wantLogin: 'yes');
          }else{
            NavigationService.pop();
            resetStateAfterDelay();
          }
        }else{
          await LocalStorageService().saveToken(verifyResponse.data?.token ?? '');
          await LocalStorageService().saveUser(user: verifyResponse.data?.user?.toJson() ?? {});
          if (loginResponse?.data?.isNewRider == true) {
            LocalStorageService().setRegistrationProgress(AppRoutes.setPassword);
            NavigationService.pushNamed(AppRoutes.setPassword);
          } else {
            LocalStorageService().setRegistrationProgress(AppRoutes.dashboard);
            ref.read(tripActivityNotifierProvider.notifier).checkTripActivity();
          }
        }



        state = AppState.success(verifyResponse);
        resetStateAfterDelay();
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class ResendOtpNotifier extends StateNotifier<AppState<ResendOtpModel>> {
  final IAuthRepo authRepo;
  final Ref ref;

  ResendOtpNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> resendOtp({required String mobile, Function(ResendOtpModel data)? onSuccess}) async {
    state = const AppState.loading();
    final result = await authRepo.resendOTP(mobile: mobile);
    result.fold(
      (failure) {
        state = AppState.error(failure);
        showNotification(message: failure.message);
      },
      (data) {
        state = AppState.success(data);
        onSuccess != null ? onSuccess(data) : null;
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class UpdatePassNotifier extends StateNotifier<AppState<CommonResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;
  UpdatePassNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> updatePassword({required String password}) async {
    state = const AppState.loading();
    final result = await authRepo.updatePassword(password: password);
    result.fold(
      (failure) {
        state = AppState.error(failure);
        showNotification(message: failure.message);
      },
      (data) {
        LocalStorageService().setRegistrationProgress(AppRoutes.setProfile);
        NavigationService.pushNamed(AppRoutes.setProfile);
        state = AppState.success(data);
        showNotification(message: data.message, isSuccess: true);
        resetStateAfterDelay();
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class ChangePasswordNotifier extends StateNotifier<AppState<CommonResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;
  ChangePasswordNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> changePassword({required String currentPassword, required String newPassword, required newConfirmPassword}) async {
    state = const AppState.loading();
    final result = await authRepo.changePassword(currentPassword: currentPassword, newPassword: newPassword, newConfirmPassword: newConfirmPassword);
    result.fold(
          (failure) {
        state = AppState.error(failure);
        showNotification(message: failure.message);
      },
          (data) {
            LocalStorageService().clearUser();
            LocalStorageService().clearToken();
        showNotification(message: data.message, isSuccess: true);
        NavigationService.pushNamedAndRemoveUntil(AppRoutes.login);
        state = AppState.success(data);
        resetStateAfterDelay();
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class ProfileUpdateNotifier extends StateNotifier<AppState<ProfileUpdateResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;

  ProfileUpdateNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> updateProfile(BuildContext context, {required Map<String, dynamic> data, bool isUpdatingProfile = false}) async {
    state = const AppState.loading();
    final response = await authRepo.profileUpdate(data: data);
    response.fold(
      (failure) {

          showNotification(message: failure.message);

        return state = AppState.error(failure);
      },
      (data) {
        LocalStorageService().saveUser(user: data.data?.user?.toJson() ?? {});
        showNotification(message: data.message, isSuccess: true);
        if(!isUpdatingProfile){
          LocalStorageService().setRegistrationProgress(AppRoutes.dashboard);
          NavigationService.pushNamedAndRemoveUntil(AppRoutes.dashboard);
        }else{
          ref.read(riderDetailsNotifierProvider.notifier).getRiderDetails();
          NavigationService.pop();
        }

        state = AppState.success(data);
        resetStateAfterDelay();
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}

class LogoutNotifier extends StateNotifier<AppState<LogoutResponse>> {
  final IAuthRepo authRepo;
  LogoutNotifier({required this.authRepo}) : super(const AppState.initial());

  Future<void> logout() async {
    state = const AppState.loading();
    final result = await authRepo.logout();
    result.fold(
      (failure) {
        state = AppState.error(failure);
        showNotification(message: failure.message);
      },
      (data) {
        LocalStorageService().clearStorage();
        NavigationService.pushNamedAndRemoveUntil(AppRoutes.login,);
        showNotification(message: data.message, isSuccess: true);
        return state = AppState.success(data);
      },
    );
  }
}

class ResetPasswordNotifier extends StateNotifier<AppState<ResetPasswordResponse>> {
  final IAuthRepo authRepo;
  final Ref ref;
  ResetPasswordNotifier({
    required this.ref,
    required this.authRepo,
  }) : super(const AppState.initial());

  Future<void> resetPassword({required Map<String, dynamic> data}) async {
    state = const AppState.loading();
    final result = await authRepo.resetPassword(data: data);
    result.fold(
      (failure) => state = AppState.error(failure),
      (data) {
        NavigationService.pushNamed(AppRoutes.login);
        state = AppState.success(data);
        resetStateAfterDelay();
      },
    );
  }

  void resetStateAfterDelay() {
    Future.delayed(Duration.zero, () {
      state = const AppState.initial();
    });
  }
}
